﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/kinesisanalytics/KinesisAnalyticsRequest.h>
#include <aws/kinesisanalytics/KinesisAnalytics_EXPORTS.h>
#include <aws/kinesisanalytics/model/InputProcessingConfiguration.h>

#include <utility>

namespace Aws {
namespace KinesisAnalytics {
namespace Model {

/**
 */
class AddApplicationInputProcessingConfigurationRequest : public KinesisAnalyticsRequest {
 public:
  AWS_KINESISANALYTICS_API AddApplicationInputProcessingConfigurationRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "AddApplicationInputProcessingConfiguration"; }

  AWS_KINESISANALYTICS_API Aws::String SerializePayload() const override;

  AWS_KINESISANALYTICS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Name of the application to which you want to add the input processing
   * configuration.</p>
   */
  inline const Aws::String& GetApplicationName() const { return m_applicationName; }
  inline bool ApplicationNameHasBeenSet() const { return m_applicationNameHasBeenSet; }
  template <typename ApplicationNameT = Aws::String>
  void SetApplicationName(ApplicationNameT&& value) {
    m_applicationNameHasBeenSet = true;
    m_applicationName = std::forward<ApplicationNameT>(value);
  }
  template <typename ApplicationNameT = Aws::String>
  AddApplicationInputProcessingConfigurationRequest& WithApplicationName(ApplicationNameT&& value) {
    SetApplicationName(std::forward<ApplicationNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Version of the application to which you want to add the input processing
   * configuration. You can use the <a
   * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_DescribeApplication.html">DescribeApplication</a>
   * operation to get the current application version. If the version specified is
   * not the current version, the <code>ConcurrentModificationException</code> is
   * returned.</p>
   */
  inline long long GetCurrentApplicationVersionId() const { return m_currentApplicationVersionId; }
  inline bool CurrentApplicationVersionIdHasBeenSet() const { return m_currentApplicationVersionIdHasBeenSet; }
  inline void SetCurrentApplicationVersionId(long long value) {
    m_currentApplicationVersionIdHasBeenSet = true;
    m_currentApplicationVersionId = value;
  }
  inline AddApplicationInputProcessingConfigurationRequest& WithCurrentApplicationVersionId(long long value) {
    SetCurrentApplicationVersionId(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the input configuration to add the input processing configuration
   * to. You can get a list of the input IDs for an application using the <a
   * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_DescribeApplication.html">DescribeApplication</a>
   * operation.</p>
   */
  inline const Aws::String& GetInputId() const { return m_inputId; }
  inline bool InputIdHasBeenSet() const { return m_inputIdHasBeenSet; }
  template <typename InputIdT = Aws::String>
  void SetInputId(InputIdT&& value) {
    m_inputIdHasBeenSet = true;
    m_inputId = std::forward<InputIdT>(value);
  }
  template <typename InputIdT = Aws::String>
  AddApplicationInputProcessingConfigurationRequest& WithInputId(InputIdT&& value) {
    SetInputId(std::forward<InputIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The <a
   * href="https://docs.aws.amazon.com/kinesisanalytics/latest/dev/API_InputProcessingConfiguration.html">InputProcessingConfiguration</a>
   * to add to the application.</p>
   */
  inline const InputProcessingConfiguration& GetInputProcessingConfiguration() const { return m_inputProcessingConfiguration; }
  inline bool InputProcessingConfigurationHasBeenSet() const { return m_inputProcessingConfigurationHasBeenSet; }
  template <typename InputProcessingConfigurationT = InputProcessingConfiguration>
  void SetInputProcessingConfiguration(InputProcessingConfigurationT&& value) {
    m_inputProcessingConfigurationHasBeenSet = true;
    m_inputProcessingConfiguration = std::forward<InputProcessingConfigurationT>(value);
  }
  template <typename InputProcessingConfigurationT = InputProcessingConfiguration>
  AddApplicationInputProcessingConfigurationRequest& WithInputProcessingConfiguration(InputProcessingConfigurationT&& value) {
    SetInputProcessingConfiguration(std::forward<InputProcessingConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_applicationName;

  long long m_currentApplicationVersionId{0};

  Aws::String m_inputId;

  InputProcessingConfiguration m_inputProcessingConfiguration;
  bool m_applicationNameHasBeenSet = false;
  bool m_currentApplicationVersionIdHasBeenSet = false;
  bool m_inputIdHasBeenSet = false;
  bool m_inputProcessingConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace KinesisAnalytics
}  // namespace Aws
