﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/pinpoint/PinpointRequest.h>
#include <aws/pinpoint/Pinpoint_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace Pinpoint {
namespace Model {

/**
 */
class ListTemplatesRequest : public PinpointRequest {
 public:
  AWS_PINPOINT_API ListTemplatesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListTemplates"; }

  AWS_PINPOINT_API Aws::String SerializePayload() const override;

  AWS_PINPOINT_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The  string that specifies which page of results to return in a paginated
   * response. This parameter is not supported for application, campaign, and journey
   * metrics.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListTemplatesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of items to include in each page of a paginated response.
   * This parameter is not supported for application, campaign, and journey
   * metrics.</p>
   */
  inline const Aws::String& GetPageSize() const { return m_pageSize; }
  inline bool PageSizeHasBeenSet() const { return m_pageSizeHasBeenSet; }
  template <typename PageSizeT = Aws::String>
  void SetPageSize(PageSizeT&& value) {
    m_pageSizeHasBeenSet = true;
    m_pageSize = std::forward<PageSizeT>(value);
  }
  template <typename PageSizeT = Aws::String>
  ListTemplatesRequest& WithPageSize(PageSizeT&& value) {
    SetPageSize(std::forward<PageSizeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The substring to match in the names of the message templates to include in
   * the results. If you specify this value, Amazon Pinpoint returns only those
   * templates whose names begin with the value that you specify.</p>
   */
  inline const Aws::String& GetPrefix() const { return m_prefix; }
  inline bool PrefixHasBeenSet() const { return m_prefixHasBeenSet; }
  template <typename PrefixT = Aws::String>
  void SetPrefix(PrefixT&& value) {
    m_prefixHasBeenSet = true;
    m_prefix = std::forward<PrefixT>(value);
  }
  template <typename PrefixT = Aws::String>
  ListTemplatesRequest& WithPrefix(PrefixT&& value) {
    SetPrefix(std::forward<PrefixT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of message template to include in the results. Valid values are:
   * EMAIL, PUSH, SMS, and VOICE. To include all types of templates in the results,
   * don't include this parameter in your request.</p>
   */
  inline const Aws::String& GetTemplateType() const { return m_templateType; }
  inline bool TemplateTypeHasBeenSet() const { return m_templateTypeHasBeenSet; }
  template <typename TemplateTypeT = Aws::String>
  void SetTemplateType(TemplateTypeT&& value) {
    m_templateTypeHasBeenSet = true;
    m_templateType = std::forward<TemplateTypeT>(value);
  }
  template <typename TemplateTypeT = Aws::String>
  ListTemplatesRequest& WithTemplateType(TemplateTypeT&& value) {
    SetTemplateType(std::forward<TemplateTypeT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_nextToken;

  Aws::String m_pageSize;

  Aws::String m_prefix;

  Aws::String m_templateType;
  bool m_nextTokenHasBeenSet = false;
  bool m_pageSizeHasBeenSet = false;
  bool m_prefixHasBeenSet = false;
  bool m_templateTypeHasBeenSet = false;
};

}  // namespace Model
}  // namespace Pinpoint
}  // namespace Aws
