﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/medical-imaging/MedicalImagingRequest.h>
#include <aws/medical-imaging/MedicalImaging_EXPORTS.h>
#include <aws/medical-imaging/model/LosslessStorageFormat.h>

#include <utility>

namespace Aws {
namespace MedicalImaging {
namespace Model {

/**
 */
class CreateDatastoreRequest : public MedicalImagingRequest {
 public:
  AWS_MEDICALIMAGING_API CreateDatastoreRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateDatastore"; }

  AWS_MEDICALIMAGING_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The data store name.</p>
   */
  inline const Aws::String& GetDatastoreName() const { return m_datastoreName; }
  inline bool DatastoreNameHasBeenSet() const { return m_datastoreNameHasBeenSet; }
  template <typename DatastoreNameT = Aws::String>
  void SetDatastoreName(DatastoreNameT&& value) {
    m_datastoreNameHasBeenSet = true;
    m_datastoreName = std::forward<DatastoreNameT>(value);
  }
  template <typename DatastoreNameT = Aws::String>
  CreateDatastoreRequest& WithDatastoreName(DatastoreNameT&& value) {
    SetDatastoreName(std::forward<DatastoreNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique identifier for API idempotency.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateDatastoreRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags provided when creating a data store.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateDatastoreRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateDatastoreRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) assigned to the Key Management Service (KMS)
   * key for accessing encrypted data.</p>
   */
  inline const Aws::String& GetKmsKeyArn() const { return m_kmsKeyArn; }
  inline bool KmsKeyArnHasBeenSet() const { return m_kmsKeyArnHasBeenSet; }
  template <typename KmsKeyArnT = Aws::String>
  void SetKmsKeyArn(KmsKeyArnT&& value) {
    m_kmsKeyArnHasBeenSet = true;
    m_kmsKeyArn = std::forward<KmsKeyArnT>(value);
  }
  template <typename KmsKeyArnT = Aws::String>
  CreateDatastoreRequest& WithKmsKeyArn(KmsKeyArnT&& value) {
    SetKmsKeyArn(std::forward<KmsKeyArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the authorizer's Lambda function.</p>
   */
  inline const Aws::String& GetLambdaAuthorizerArn() const { return m_lambdaAuthorizerArn; }
  inline bool LambdaAuthorizerArnHasBeenSet() const { return m_lambdaAuthorizerArnHasBeenSet; }
  template <typename LambdaAuthorizerArnT = Aws::String>
  void SetLambdaAuthorizerArn(LambdaAuthorizerArnT&& value) {
    m_lambdaAuthorizerArnHasBeenSet = true;
    m_lambdaAuthorizerArn = std::forward<LambdaAuthorizerArnT>(value);
  }
  template <typename LambdaAuthorizerArnT = Aws::String>
  CreateDatastoreRequest& WithLambdaAuthorizerArn(LambdaAuthorizerArnT&& value) {
    SetLambdaAuthorizerArn(std::forward<LambdaAuthorizerArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The lossless storage format for the datastore.</p>
   */
  inline LosslessStorageFormat GetLosslessStorageFormat() const { return m_losslessStorageFormat; }
  inline bool LosslessStorageFormatHasBeenSet() const { return m_losslessStorageFormatHasBeenSet; }
  inline void SetLosslessStorageFormat(LosslessStorageFormat value) {
    m_losslessStorageFormatHasBeenSet = true;
    m_losslessStorageFormat = value;
  }
  inline CreateDatastoreRequest& WithLosslessStorageFormat(LosslessStorageFormat value) {
    SetLosslessStorageFormat(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_datastoreName;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::Map<Aws::String, Aws::String> m_tags;

  Aws::String m_kmsKeyArn;

  Aws::String m_lambdaAuthorizerArn;

  LosslessStorageFormat m_losslessStorageFormat{LosslessStorageFormat::NOT_SET};
  bool m_datastoreNameHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
  bool m_tagsHasBeenSet = false;
  bool m_kmsKeyArnHasBeenSet = false;
  bool m_lambdaAuthorizerArnHasBeenSet = false;
  bool m_losslessStorageFormatHasBeenSet = false;
};

}  // namespace Model
}  // namespace MedicalImaging
}  // namespace Aws
