﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/databrew/GlueDataBrewRequest.h>
#include <aws/databrew/GlueDataBrew_EXPORTS.h>
#include <aws/databrew/model/DataCatalogOutput.h>
#include <aws/databrew/model/DatabaseOutput.h>
#include <aws/databrew/model/EncryptionMode.h>
#include <aws/databrew/model/LogSubscription.h>
#include <aws/databrew/model/Output.h>
#include <aws/databrew/model/RecipeReference.h>

#include <utility>

namespace Aws {
namespace GlueDataBrew {
namespace Model {

/**
 */
class CreateRecipeJobRequest : public GlueDataBrewRequest {
 public:
  AWS_GLUEDATABREW_API CreateRecipeJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateRecipeJob"; }

  AWS_GLUEDATABREW_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the dataset that this job processes.</p>
   */
  inline const Aws::String& GetDatasetName() const { return m_datasetName; }
  inline bool DatasetNameHasBeenSet() const { return m_datasetNameHasBeenSet; }
  template <typename DatasetNameT = Aws::String>
  void SetDatasetName(DatasetNameT&& value) {
    m_datasetNameHasBeenSet = true;
    m_datasetName = std::forward<DatasetNameT>(value);
  }
  template <typename DatasetNameT = Aws::String>
  CreateRecipeJobRequest& WithDatasetName(DatasetNameT&& value) {
    SetDatasetName(std::forward<DatasetNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of an encryption key that is used to protect
   * the job.</p>
   */
  inline const Aws::String& GetEncryptionKeyArn() const { return m_encryptionKeyArn; }
  inline bool EncryptionKeyArnHasBeenSet() const { return m_encryptionKeyArnHasBeenSet; }
  template <typename EncryptionKeyArnT = Aws::String>
  void SetEncryptionKeyArn(EncryptionKeyArnT&& value) {
    m_encryptionKeyArnHasBeenSet = true;
    m_encryptionKeyArn = std::forward<EncryptionKeyArnT>(value);
  }
  template <typename EncryptionKeyArnT = Aws::String>
  CreateRecipeJobRequest& WithEncryptionKeyArn(EncryptionKeyArnT&& value) {
    SetEncryptionKeyArn(std::forward<EncryptionKeyArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The encryption mode for the job, which can be one of the following:</p> <ul>
   * <li> <p> <code>SSE-KMS</code> - Server-side encryption with keys managed by
   * KMS.</p> </li> <li> <p> <code>SSE-S3</code> - Server-side encryption with keys
   * managed by Amazon S3.</p> </li> </ul>
   */
  inline EncryptionMode GetEncryptionMode() const { return m_encryptionMode; }
  inline bool EncryptionModeHasBeenSet() const { return m_encryptionModeHasBeenSet; }
  inline void SetEncryptionMode(EncryptionMode value) {
    m_encryptionModeHasBeenSet = true;
    m_encryptionMode = value;
  }
  inline CreateRecipeJobRequest& WithEncryptionMode(EncryptionMode value) {
    SetEncryptionMode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique name for the job. Valid characters are alphanumeric (A-Z, a-z, 0-9),
   * hyphen (-), period (.), and space.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateRecipeJobRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Enables or disables Amazon CloudWatch logging for the job. If logging is
   * enabled, CloudWatch writes one log stream for each job run.</p>
   */
  inline LogSubscription GetLogSubscription() const { return m_logSubscription; }
  inline bool LogSubscriptionHasBeenSet() const { return m_logSubscriptionHasBeenSet; }
  inline void SetLogSubscription(LogSubscription value) {
    m_logSubscriptionHasBeenSet = true;
    m_logSubscription = value;
  }
  inline CreateRecipeJobRequest& WithLogSubscription(LogSubscription value) {
    SetLogSubscription(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of nodes that DataBrew can consume when the job processes
   * data.</p>
   */
  inline int GetMaxCapacity() const { return m_maxCapacity; }
  inline bool MaxCapacityHasBeenSet() const { return m_maxCapacityHasBeenSet; }
  inline void SetMaxCapacity(int value) {
    m_maxCapacityHasBeenSet = true;
    m_maxCapacity = value;
  }
  inline CreateRecipeJobRequest& WithMaxCapacity(int value) {
    SetMaxCapacity(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of times to retry the job after a job run fails.</p>
   */
  inline int GetMaxRetries() const { return m_maxRetries; }
  inline bool MaxRetriesHasBeenSet() const { return m_maxRetriesHasBeenSet; }
  inline void SetMaxRetries(int value) {
    m_maxRetriesHasBeenSet = true;
    m_maxRetries = value;
  }
  inline CreateRecipeJobRequest& WithMaxRetries(int value) {
    SetMaxRetries(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more artifacts that represent the output from running the job.</p>
   */
  inline const Aws::Vector<Output>& GetOutputs() const { return m_outputs; }
  inline bool OutputsHasBeenSet() const { return m_outputsHasBeenSet; }
  template <typename OutputsT = Aws::Vector<Output>>
  void SetOutputs(OutputsT&& value) {
    m_outputsHasBeenSet = true;
    m_outputs = std::forward<OutputsT>(value);
  }
  template <typename OutputsT = Aws::Vector<Output>>
  CreateRecipeJobRequest& WithOutputs(OutputsT&& value) {
    SetOutputs(std::forward<OutputsT>(value));
    return *this;
  }
  template <typename OutputsT = Output>
  CreateRecipeJobRequest& AddOutputs(OutputsT&& value) {
    m_outputsHasBeenSet = true;
    m_outputs.emplace_back(std::forward<OutputsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>One or more artifacts that represent the Glue Data Catalog output from
   * running the job.</p>
   */
  inline const Aws::Vector<DataCatalogOutput>& GetDataCatalogOutputs() const { return m_dataCatalogOutputs; }
  inline bool DataCatalogOutputsHasBeenSet() const { return m_dataCatalogOutputsHasBeenSet; }
  template <typename DataCatalogOutputsT = Aws::Vector<DataCatalogOutput>>
  void SetDataCatalogOutputs(DataCatalogOutputsT&& value) {
    m_dataCatalogOutputsHasBeenSet = true;
    m_dataCatalogOutputs = std::forward<DataCatalogOutputsT>(value);
  }
  template <typename DataCatalogOutputsT = Aws::Vector<DataCatalogOutput>>
  CreateRecipeJobRequest& WithDataCatalogOutputs(DataCatalogOutputsT&& value) {
    SetDataCatalogOutputs(std::forward<DataCatalogOutputsT>(value));
    return *this;
  }
  template <typename DataCatalogOutputsT = DataCatalogOutput>
  CreateRecipeJobRequest& AddDataCatalogOutputs(DataCatalogOutputsT&& value) {
    m_dataCatalogOutputsHasBeenSet = true;
    m_dataCatalogOutputs.emplace_back(std::forward<DataCatalogOutputsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Represents a list of JDBC database output objects which defines the output
   * destination for a DataBrew recipe job to write to. </p>
   */
  inline const Aws::Vector<DatabaseOutput>& GetDatabaseOutputs() const { return m_databaseOutputs; }
  inline bool DatabaseOutputsHasBeenSet() const { return m_databaseOutputsHasBeenSet; }
  template <typename DatabaseOutputsT = Aws::Vector<DatabaseOutput>>
  void SetDatabaseOutputs(DatabaseOutputsT&& value) {
    m_databaseOutputsHasBeenSet = true;
    m_databaseOutputs = std::forward<DatabaseOutputsT>(value);
  }
  template <typename DatabaseOutputsT = Aws::Vector<DatabaseOutput>>
  CreateRecipeJobRequest& WithDatabaseOutputs(DatabaseOutputsT&& value) {
    SetDatabaseOutputs(std::forward<DatabaseOutputsT>(value));
    return *this;
  }
  template <typename DatabaseOutputsT = DatabaseOutput>
  CreateRecipeJobRequest& AddDatabaseOutputs(DatabaseOutputsT&& value) {
    m_databaseOutputsHasBeenSet = true;
    m_databaseOutputs.emplace_back(std::forward<DatabaseOutputsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Either the name of an existing project, or a combination of a recipe and a
   * dataset to associate with the recipe.</p>
   */
  inline const Aws::String& GetProjectName() const { return m_projectName; }
  inline bool ProjectNameHasBeenSet() const { return m_projectNameHasBeenSet; }
  template <typename ProjectNameT = Aws::String>
  void SetProjectName(ProjectNameT&& value) {
    m_projectNameHasBeenSet = true;
    m_projectName = std::forward<ProjectNameT>(value);
  }
  template <typename ProjectNameT = Aws::String>
  CreateRecipeJobRequest& WithProjectName(ProjectNameT&& value) {
    SetProjectName(std::forward<ProjectNameT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const RecipeReference& GetRecipeReference() const { return m_recipeReference; }
  inline bool RecipeReferenceHasBeenSet() const { return m_recipeReferenceHasBeenSet; }
  template <typename RecipeReferenceT = RecipeReference>
  void SetRecipeReference(RecipeReferenceT&& value) {
    m_recipeReferenceHasBeenSet = true;
    m_recipeReference = std::forward<RecipeReferenceT>(value);
  }
  template <typename RecipeReferenceT = RecipeReference>
  CreateRecipeJobRequest& WithRecipeReference(RecipeReferenceT&& value) {
    SetRecipeReference(std::forward<RecipeReferenceT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the Identity and Access Management (IAM)
   * role to be assumed when DataBrew runs the job.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  CreateRecipeJobRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Metadata tags to apply to this job.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateRecipeJobRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateRecipeJobRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The job's timeout in minutes. A job that attempts to run longer than this
   * timeout period ends with a status of <code>TIMEOUT</code>.</p>
   */
  inline int GetTimeout() const { return m_timeout; }
  inline bool TimeoutHasBeenSet() const { return m_timeoutHasBeenSet; }
  inline void SetTimeout(int value) {
    m_timeoutHasBeenSet = true;
    m_timeout = value;
  }
  inline CreateRecipeJobRequest& WithTimeout(int value) {
    SetTimeout(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_datasetName;

  Aws::String m_encryptionKeyArn;

  EncryptionMode m_encryptionMode{EncryptionMode::NOT_SET};

  Aws::String m_name;

  LogSubscription m_logSubscription{LogSubscription::NOT_SET};

  int m_maxCapacity{0};

  int m_maxRetries{0};

  Aws::Vector<Output> m_outputs;

  Aws::Vector<DataCatalogOutput> m_dataCatalogOutputs;

  Aws::Vector<DatabaseOutput> m_databaseOutputs;

  Aws::String m_projectName;

  RecipeReference m_recipeReference;

  Aws::String m_roleArn;

  Aws::Map<Aws::String, Aws::String> m_tags;

  int m_timeout{0};
  bool m_datasetNameHasBeenSet = false;
  bool m_encryptionKeyArnHasBeenSet = false;
  bool m_encryptionModeHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_logSubscriptionHasBeenSet = false;
  bool m_maxCapacityHasBeenSet = false;
  bool m_maxRetriesHasBeenSet = false;
  bool m_outputsHasBeenSet = false;
  bool m_dataCatalogOutputsHasBeenSet = false;
  bool m_databaseOutputsHasBeenSet = false;
  bool m_projectNameHasBeenSet = false;
  bool m_recipeReferenceHasBeenSet = false;
  bool m_roleArnHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_timeoutHasBeenSet = false;
};

}  // namespace Model
}  // namespace GlueDataBrew
}  // namespace Aws
