#!/usr/bin/env python3
"""
Spheres on two hexagonal close packed layers
"""
import bornagain as ba
from bornagain import ba_plot as bp, deg, nm, R3


def get_sample():
    """
    A sample with spheres on a substrate,
    forming two hexagonal close packed layers.
    """

    # Materials
    material_particle = ba.RefractiveMaterial("Particle", 0.0006, 2e-08)
    material_substrate = ba.RefractiveMaterial("Substrate", 6e-06, 2e-08)
    vacuum = ba.RefractiveMaterial("Vacuum", 0, 0)

    # Particles
    ff = ba.Sphere(10*nm)
    particle_1 = ba.Particle(material_particle, ff)
    particle_2 = ba.Particle(material_particle, ff)
    particle_2_position = R3(10*nm, 10*nm, 17.3205080757*nm)
    particle_2.translate(particle_2_position)

    # Lattice basis as compound particle
    basis = ba.Compound()
    basis.addComponent(particle_1)
    basis.addComponent(particle_2)

    # 2D lattice
    lattice = ba.HexagonalLattice2D(20*nm, 0)
    iff = ba.Interference2DLattice(lattice)
    profile = ba.Profile2DCauchy(10*nm, 10*nm, 0)
    iff.setDecayFunction(profile)

    # Particle layouts
    layout = ba.ParticleLayout()
    layout.addParticle(basis)
    layout.setInterference(iff)
    layout.setTotalParticleSurfaceDensity(0.00288675134595)

    # Layers
    layer_1 = ba.Layer(vacuum)
    layer_1.addLayout(layout)
    layer_2 = ba.Layer(material_substrate)

    # Sample
    sample = ba.Sample()
    sample.addLayer(layer_1)
    sample.addLayer(layer_2)

    return sample


def get_simulation(sample):
    beam = ba.Beam(1e9, 0.1*nm, 0.2*deg)
    n = <%= test_mode ? 11 : 200 %>
    detector = ba.SphericalDetector(n, -1*deg, 1*deg, n, 0, 1*deg)
    simulation = ba.ScatteringSimulation(beam, sample, detector)
    return simulation


if __name__ == '__main__':
    sample = get_sample()
    simulation = get_simulation(sample)
    result = simulation.simulate()
    <%- if test_mode -%>
    from bornagain import ba_check
    ba_check.persistence_test(result)
    <%- elsif figure_mode -%>
    plotargs = bp.parse_commandline()
    bp.plot_datafield(result, **plotargs)
    bp.export(**plotargs)
    <%- else -%>
    bp.plot_datafield(result)
    bp.plt.show()
    <%- end -%>
