﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/resource-explorer-2/ResourceExplorer2Request.h>
#include <aws/resource-explorer-2/ResourceExplorer2_EXPORTS.h>
#include <aws/resource-explorer-2/model/IndexType.h>

#include <utility>

namespace Aws {
namespace ResourceExplorer2 {
namespace Model {

/**
 */
class ListIndexesRequest : public ResourceExplorer2Request {
 public:
  AWS_RESOURCEEXPLORER2_API ListIndexesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListIndexes"; }

  AWS_RESOURCEEXPLORER2_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>If specified, limits the output to only indexes of the specified Type, either
   * <code>LOCAL</code> or <code>AGGREGATOR</code>.</p> <p>Use this option to
   * discover the aggregator index for your account.</p>
   */
  inline IndexType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(IndexType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline ListIndexesRequest& WithType(IndexType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If specified, limits the response to only information about the index in the
   * specified list of Amazon Web Services Regions.</p>
   */
  inline const Aws::Vector<Aws::String>& GetRegions() const { return m_regions; }
  inline bool RegionsHasBeenSet() const { return m_regionsHasBeenSet; }
  template <typename RegionsT = Aws::Vector<Aws::String>>
  void SetRegions(RegionsT&& value) {
    m_regionsHasBeenSet = true;
    m_regions = std::forward<RegionsT>(value);
  }
  template <typename RegionsT = Aws::Vector<Aws::String>>
  ListIndexesRequest& WithRegions(RegionsT&& value) {
    SetRegions(std::forward<RegionsT>(value));
    return *this;
  }
  template <typename RegionsT = Aws::String>
  ListIndexesRequest& AddRegions(RegionsT&& value) {
    m_regionsHasBeenSet = true;
    m_regions.emplace_back(std::forward<RegionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of results that you want included on each page of the
   * response. If you do not include this parameter, it defaults to a value
   * appropriate to the operation. If additional items exist beyond those included in
   * the current response, the <code>NextToken</code> response element is present and
   * has a value (is not null). Include that value as the <code>NextToken</code>
   * request parameter in the next call to the operation to get the next part of the
   * results.</p>  <p>An API operation can return fewer results than the
   * maximum even when there are more results available. You should check
   * <code>NextToken</code> after every operation to ensure that you receive all of
   * the results.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListIndexesRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The parameter for receiving additional results if you receive a
   * <code>NextToken</code> response in a previous request. A <code>NextToken</code>
   * response indicates that more output is available. Set this parameter to the
   * value of the previous call's <code>NextToken</code> response to indicate where
   * the output should continue from. The pagination tokens expire after 24
   * hours.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListIndexesRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}
 private:
  IndexType m_type{IndexType::NOT_SET};

  Aws::Vector<Aws::String> m_regions;

  int m_maxResults{0};

  Aws::String m_nextToken;
  bool m_typeHasBeenSet = false;
  bool m_regionsHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
};

}  // namespace Model
}  // namespace ResourceExplorer2
}  // namespace Aws
